# Config
A perl 6 library for reading and writing configuration files.

[![Build Status](https://travis-ci.org/scriptkitties/p6-Config.svg?branch=master)](https://travis-ci.org/scriptkitties/p6-Config)

## Installation
This module can be installed using `zef`:

```
zef install Config
```

Depending on the type of configuration file you want to work on, you will need a
`Config::Parser::` module as well. If you just want an easy-to-use configuration
object without reading/writing a file, no parser is needed.

## Usage
Include the `Config` module in your script, instantiate it and use it as you
please.

```perl6
use Config;

my Config $config = Config.new();

# Load a simple configuration hash
$config.read({
    keyOne => "value",
    keyTwo => {
        NestedKey => "other value"
    }
});

# Load a configuration files
$config.read("/etc/config.yaml");

# Load a configuration file with a specific parser
$config.read("/etc/config", "Config::Parser::ini");

# Retrieve a simple key
$config.get("keyOne");

# As of v1.2.0, `Config` support associative indexing:
$config<keyOne>;

# Retrieve a nested key
$config.get("keyTwo.NestedKey");

# Write out the configuration file
$config.write("/etc/config.yaml");

# Write out the configuration in another format
$config.write("/etc/config.json", "Config::Parser::json");
```

### Available parsers
Because there's so many ways to structure your configuration files, the parsers
for these are their own modules. This allows for easy implementing new parsers,
or providing a custom parser for your project's configuration file.

The parser will be loaded during runtime, but you have to make sure it is
installed yourself.

The following parsers are available:

- [`Config::Parser::json`](https://github.com/robertlemmen/perl6-config-json)
- [`Config::Parser::toml`](https://github.com/scriptkitties/p6-Config-Parser-toml)
- [`Config::Parser::yaml`](https://github.com/scriptkitties/p6-Config-Parser-yaml)

### Writing your own parser
If you want to make your own parser, simply make a new class which extends the
`Config::Parser` class, and implements the `read` and `write` methods. The
`read` method *must* return a `Hash`. The `write` method *must* return a
`Bool`, `True` when writing was successful, `False` if not. Throwing
`Exception`s to indicate the kind of failure is recommended.

## License
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
