#!/usr/bin/env perl
use strict;
use warnings;

use Test::More tests => 14;
use Test::MockModule;
use Test::Deep;
use Test::Exception;

use Git::Wrapper::More;

VARIOUS_TESTS_USING_SHOW:
{
    my $mgw = Test::MockModule->new('Git::Wrapper');

    my @dummy_commit = (
        'commit 1feadfcc5f5411fbe8128a1a2f06f17a2208503c',
        'Author: Random User <user@cp.net>',
        'Date:   Tue Jun 25 09:50:49 2019 -0500',
        '', '    Commit message', '',
    );

    $mgw->mock(
        'show',
        sub {
            my $self = shift;
            return @dummy_commit;
        },
    );

    my $gwm = Git::Wrapper::More->new( dir => '/dummy/dir', verbose => 1 );

    my $cmd = sub {
        $gwm->{gw_obj}->show( { summary => 1 }, '29832838328303092900' );
    };

    {
        my $label = 'Happy Path';

        my @ret = $gwm->_run_cmd( $cmd, 'array' );
        cmp_deeply( \@ret, \@dummy_commit, "$label: _run_cmd returns proper output from a simple command" );
    }

    {
        my $label = 'Scalar';

        my @ret = $gwm->_run_cmd( $cmd, 'scalar' );

        is( $ret[0], 6, "$label: _run_cmd predictably converted array to one number" );
    }

    # reference type must be array or scalar, or die
    my @throw_tests = (
        { 'Optional ref_type input - wrong value'                     => 'other' },
        { 'Optional ref_type input - wrong value leading underscore'  => '_array' },
        { 'Optional ref_type input - wrong value trailing underscore' => 'array_' },
        { 'Optional ref_type input - wrong value leading underscore'  => '_scalar' },
        { 'Optional ref_type input - wrong value trailing underscore' => 'scalar_' },
        { 'Optional ref_type input - empty value' => '' },
    );

    foreach my $throw_test (@throw_tests) {
        my $label = ( keys %{$throw_test} )[0];

        throws_ok(
            sub { $gwm->_run_cmd( $cmd, $$throw_test{$label} ) },
            qr/invalid value for \$ref_type/,
            "$label: dies with proper error"
        );
    }

    my @lives_tests = (
        { 'Optional ref_type input - array' => 'array' },
        { 'Optional ref_type input - undef' => undef },
    );

    foreach my $lives_test (@lives_tests) {
        my $label = ( keys %{$lives_test} )[0];

        lives_ok(
            sub { $gwm->_run_cmd( $cmd, $$lives_test{$label} ) },
            "Lives: $label"
        );
    }
}

GIT_WRAPPER_DEATH_RETURN_VALS:
{
    my $label = 'Git::Wrapper dies and returns error';

    close(STDERR);
    my $hide_stderr;
    open( STDERR, ">>", \$hide_stderr );

    my $mgw2 = Test::MockModule->new('Git::Wrapper');
    my $gwm2 = Git::Wrapper::More->new( dir => '/dummy/dir', verbose => 1 );

    $mgw2->mock( 'error',      sub { return 'from mocked error' } );
    $mgw2->mock( 'merge_base', sub { die $gwm2->{gw_obj}->error } );

    my $cmd = sub {
        ( $gwm2->{gw_obj}->merge_base( "nonexistant", "nonexistant2" ) )[0];
    };

    lives_ok { $gwm2->_run_cmd($cmd) } "$label: Git::Wrapper::More lives";
    like( $hide_stderr, qr/from mocked error/, "$label: Git::Wrapper::More returns error object message" );
}

{
    my $label = 'Git::Wrapper dies and returns output';

    close(STDERR);
    my $hide_stderr;
    open( STDERR, ">>", \$hide_stderr );

    my $mgw2 = Test::MockModule->new('Git::Wrapper');
    my $gwm2 = Git::Wrapper::More->new( dir => '/dummy/dir', verbose => 1 );

    $mgw2->mock( 'output',     sub { return 'from mocked output' } );
    $mgw2->mock( 'merge_base', sub { die $gwm2->{gw_obj}->output } );

    my $cmd = sub {
        ( $gwm2->{gw_obj}->merge_base( "nonexistant", "nonexistant2" ) )[0];
    };

    lives_ok { $gwm2->_run_cmd($cmd) } "$label: Git::Wrapper::More lives";
    like( $hide_stderr, qr/from mocked output/, "$label: Git::Wrapper::More returns error object output" );
}

__END__
