package WARC::Record;						# -*- CPerl -*-

use strict;
use warnings;

our @ISA = qw();

require WARC; *WARC::Record::VERSION = \$WARC::VERSION;

use overload '<=>' => 'compareTo';

=head1 NAME

WARC::Record - one record from a WARC file

=head1 SYNOPSIS

  use WARC;		# or ...
  use WARC::Volume;	# or ...
  use WARC::Collection;

  # WARC::Record objects are returned from ->record_at and ->search methods

  # Construct a record, as when preparing a WARC file
  $warcinfo = new WARC::Record (type => 'warcinfo');

  # Accessors

  $value = $record->field($name);

  $version = $record->protocol;	# analogous to HTTP::Message::protocol
  $volume = $record->volume;
  $offset = $record->offset;
  $record = $record->next;

  $fields = $record->fields;

...

=head1 DESCRIPTION

C<WARC::Record> objects come in two flavors with a common interface.
Records read from WARC files are read-only and have meaningful return
values from the methods listed in "Methods on records from WARC files".
Records constructed in memory can be updated and those same methods all
return undef.

=head2 Common Methods

=over

=item $record-E<gt>fields

Get the internal C<WARC::Fields> object that contains WARC record headers.

=cut

sub fields {
}

=item $record-E<gt>field( $name )

Get the value of the WARC header named $name from the internal
C<WARC::Fields> object.

=cut

sub field {
}

=item $record E<lt>=E<gt> $other_record

=item $record-E<gt>compareTo( $other_record )

Compare two C<WARC::Record> objects according to a simple total order:
ordering by starting offset for two records in the same file, and by
filename of the containing C<WARC::Volume> objects for records in different
files.

Perl constructs a C<==> operator using this method, so WARC record objects
will compare as equal iff they refer to the same physical record.

=cut

sub compareTo {
}


=back

=head2 Methods on records from WARC files

These methods all return undef if called on a C<WARC::Record> object that
does not represent a record in a WARC file.

=over

=item $record-E<gt>protocol

Return the format and version tag for this record.  For WARC 1.0, this
method returns 'WARC/1.0'.

=cut

sub protocol { return undef }

=item $record-E<gt>volume

Return the C<WARC::Volume> object representing the file in which this
record is located.

=cut

sub volume { return undef }

=item $record-E<gt>offset

Return the file offset at which this record can be found.

=cut

sub offset { return undef }

=item $record-E<gt>next

Return the next C<WARC::Record> in the WARC file that contains this record.

=cut

sub next { return undef }

=item $record-E<gt>replay

=item $record-E<gt>replay( as =E<gt> $type )

Return a protocol-specific object representing the record contents.

This method returns undef if the library does not recognize the protocol
message stored in the record and croaks if a requested conversion is not
supported.

A record with Content-Type "application/http" with an appropriate "msgtype"
parameter produces an C<HTTP::Request> or C<HTTP::Response> object.  An
unknown "msgtype" on "application/http" produces a generic
C<HTTP::Message>.  The returned object may be a subclass to support
deferred loading of entity bodies.

A request to replay a record "as =E<gt> http" attempts to convert whatever
is stored in the record to an HTTP exchange, analogous to the "everything
is HTTP" interface that C<LWP> provides.

=cut

sub replay { return undef }

=item $record-E<gt>open_block

Return a tied filehandle that reads the WARC record block.

The WARC record block is the content of a WARC record, analogous to the
entity body in an C<HTTP::Message>.

=cut

sub open_block { return undef }

=item $record-E<gt>open_payload

Return a tied filehandle that reads the WARC record payload.

The WARC record payload is defined as the decoded content of the protocol
response or other resource stored in the record.  This method returns undef
if called on a WARC record that has no payload or content that we do not
recognize.

=cut

sub open_payload { return undef }

=back

=head2 Methods on fresh WARC records

=over

=item $record = new WARC::Record (I<key> =E<gt> I<value>, ...)

Construct a fresh WARC record, suitable for use with C<WARC::Builder>.

=cut

sub new {
}

=back

=cut

1;
__END__

=head1 AUTHOR

Jacob Bachmeyer, E<lt>jcb@cpan.orgE<gt>

=head1 SEE ALSO

L<WARC>, L<HTTP::Message>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2019 by Jacob Bachmeyer

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
